<?php


namespace App\Services;

use App\Models\Translation;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Cache;

class TranslationService
{
    protected $targetLanguages = ['en', 'fr'];

    public function extractAndTranslateAllAdminPages()
    {

        // دریافت تمام مسیرهای ادمین
        $adminRoutes = collect(\Route::getRoutes())
            ->filter(function ($route) {
                // بررسی هم prefix و هم نام مسیر و هم URI
                $isAdmin = false;

                if (isset($route->action['prefix'])) {
                    $isAdmin = Str::startsWith($route->action['prefix'], 'admin');
                }

                if (!$isAdmin && isset($route->action['as'])) {
                    $isAdmin = Str::startsWith($route->action['as'], 'admin.');
                }

                if (!$isAdmin) {
                    $isAdmin = Str::startsWith($route->uri(), 'admin/');
                }

                return $isAdmin && in_array('GET', $route->methods());
            })
            ->map(function ($route) {
                return $route->uri();
            })
            ->unique()
            ->values();

        $allTexts = [];

        foreach ($adminRoutes as $route) {
            try {
                $response = app()->handle(
                    \Illuminate\Http\Request::create($route)
                );

                if ($response->isSuccessful()) {
                    $content = $response->getContent();
                    if (!empty($content)) {
                        $texts = $this->extractAdminTexts($content);
                        $allTexts = array_merge($allTexts, $texts);
                    }
                }
            } catch (\Exception $e) {
                continue;
            }
        }

        $allTexts = array_unique(array_filter($allTexts, [$this, 'isValidText']));
        foreach ($allTexts as $text) {
            $translation = Translation::findOrCreate($text);

            foreach ($this->targetLanguages as $lang) {
                if (!isset($translation->translations[$lang])) {
                    $translated = $this->translateText($text, 'fa', $lang);
                    $translation->addTranslation($lang, $translated);
                }
            }
        }

        return count($allTexts);
    }

    public function extractAdminTexts($html)
    {
        $dom = new \DOMDocument();
        libxml_use_internal_errors(true);
        $html = mb_convert_encoding($html, 'HTML-ENTITIES', 'UTF-8');
        @$dom->loadHTML($html);
        libxml_clear_errors();

        $xpath = new \DOMXPath($dom);
        $texts = [];

        // انتخاب تگ‌های مورد نظر در ادمین
        $query = '//text()[(parent::label or parent::a or parent::button or parent::span or '.
            'parent::small or parent::strong or parent::p or '.
            'parent::h1 or parent::h2 or parent::h3 or parent::h4 or parent::h5 or parent::h6 or parent::th) '.
            'or parent::*[@data-original-title or @placeholder or @data-placeholder or @value]]';

        $nodes = $xpath->query($query);

        foreach ($nodes as $node) {
            // پردازش متن اصلی
            $text = trim($node->nodeValue);
            if ($this->isValidText($text)) {
                $texts[] = $text;
            }

            $parent = $node->parentNode;

            // پردازش attributeهای مختلف
            $attributesToCheck = [
                'data-original-title',
                'placeholder',
                'data-placeholder',
                'value'
            ];

            foreach ($attributesToCheck as $attr) {
                if ($parent->hasAttribute($attr)) {
                    $attrText = trim($parent->getAttribute($attr));
                    if ($this->isValidText($attrText)) {
                        $texts[] = $attrText;
                    }
                }
            }

            // پردازش خاص برای selectهای خاص
            if ($parent instanceof \DOMElement && $parent->tagName === 'select') {
                // پردازش optionهای select با id="status"
                if ($parent->getAttribute('id') === 'status') {
                    foreach ($parent->getElementsByTagName('option') as $option) {
                        $optionText = trim($option->nodeValue);
                        if ($this->isValidText($optionText)) {
                            $texts[] = $optionText;
                        }
                    }
                }
            }
        }

        // حذف مقادیر تکراری
        $texts = array_unique($texts);

        return array_values($texts);
    }

    public function isValidText($text)
    {
        if (empty($text) || strlen($text) < 2) return false;
        if (is_numeric($text)) return false;
        if (preg_match('/^[^\p{L}\p{N}]+$/u', $text)) return false;
        if (preg_match('/^\d+$/', $text)) return false;
        if (preg_match('/^[a-zA-Z0-9_\-]+$/', $text)) return false;

        return true;
    }


    public function translateText($text, $source, $target)
    {
        // استفاده از سرویس ترجمه گوگل
        try {
            // ایجاد یک کلید منحصر به فرد برای کش بر اساس متن و زبان هدف
            $cacheKey = 'translation_' . md5($text . '_' . $target);

            // بررسی وجود نتیجه در کش
            if (Cache::has($cacheKey)) {
                return Cache::get($cacheKey);
            }

            // ایجاد URL برای API با تشخیص خودکار زبان منبع
            $url = 'https://translate.googleapis.com/translate_a/single?client=gtx&sl=auto&tl=' . $target . '&dt=t&q=' . urlencode($text);

            // استفاده از cURL برای ارسال درخواست
            $ch = curl_init($url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            $response = curl_exec($ch);
            curl_close($ch);

            // تبدیل پاسخ به آرایه
            $responseArray = json_decode($response, true);

            // استخراج ترجمه
            if (isset($responseArray[0][0][0])) {
                $translatedText = $responseArray[0][0][0];

                // ذخیره نتیجه ترجمه در کش به مدت 24 ساعت (86400 ثانیه)
                Cache::put($cacheKey, $translatedText, 86400);

                return $translatedText;
            } else {
                return $text;
            }

        } catch (\Exception $e) {
            return $text; // در صورت خطا متن اصلی بازگردانده شود
        }
    }
}
