<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class Course extends Model
{
    use HasFactory;

    protected $guarded = ['id', 'created_at', 'updated_at'];

    public function langs()
    {
        return $this->morphMany('App\Models\Lang', 'langs');
    }

    // زبان فعال بر اساس وضعیت
    public function lang_active()
    {
        return $this->morphOne('App\Models\Lang', 'langs')
            ->where('lang', app()->getLocale())
            ->where('col_name', 'status')
            ->where('text', 'active');
    }

    // Scope برای فیلتر کردن بر اساس وضعیت و زبان
    public function scopeCheckStatus($query)
    {
        if (default_lang() == lang()) {
            return $query->where('status', 'active');
        } else {
            return $query->whereHas('lang_active');
        }
    }

    protected static function boot()
    {
        parent::boot();
        static::deleting(function ($item) {

            if(count($item->langs))
            {
                foreach ($item->langs as $lang)
                {
                    $lang->delete();
                }
            }

        });
    }

    public function groups()
    {
        return $this->hasMany(CourseGroup::class);
    }

    public function instructors()
    {
        return $this->belongsToMany(User::class,'course_instructors');
    }

    public function chapters()
    {
        return $this->hasMany(CourseChapter::class, 'course_id');
    }

    public function reviews()
    {
        return $this->hasMany(CourseReview::class);
    }

    public function enrollments()
    {
        return $this->hasMany(CourseEnrollment::class);
    }

    public function certificates()
    {
        return $this->hasOne(CourseCertificate::class);
    }

    public function photo()
    {
        return $this->morphOne('App\Models\Photo', 'pictures')->where('status', 'active')->where('type', 'photo');
    }

    public function video()
    {
        return $this->morphOne('App\Models\Photo', 'pictures')->where('status', 'active')->where('type', 'video');
    }

    public function contest()
    {
        return $this->belongsTo(Contest::class);
    }

    public function grade()
    {
        return $this->belongsTo(Grade::class, 'grades_id');
    }

    public function category()
    {
        return $this->belongsTo(CategoryFaq::class, 'category_id');
    }

    /**
     * تاریخ پایان دوره را بر اساس start_date و duration_weeks محاسبه می‌کند.
     * @return Carbon|null
     */

    public function endDate()
    {
        if (!$this->start_date || !$this->duration_weeks) {
            return null;
        }

        $weeks = (int) filter_var($this->duration_weeks, FILTER_SANITIZE_NUMBER_INT);
        return Carbon::parse($this->start_date)->addWeeks($weeks)->subDay();
    }

    public function isActive()
    {
        $endDate = $this->endDate();
        return !$endDate || $endDate >= now();
    }


}
