<?php

namespace App\Http\Controllers\Front;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Services\SportsEventsService;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Pagination\Paginator;
use Carbon\Carbon; // ← اضافه شده برای حل خطا

class SeitController extends Controller
{
    protected $service;

    public function __construct(SportsEventsService $service)
    {
        $this->service = $service;
    }

    public function index(Request $request)
    {
        return view('front.index');
    }

    public function ticket(Request $request)
    {
        $countries = $this->service->getCountries();
        $countries = $this->service->filterCountries($countries['data'] ?? []);

        $filters = [];
        if ($request->filled('country_id')) $filters['countryId'] = $request->country_id;
        if ($request->filled('city_id')) $filters['cityId'] = $request->city_id;
        if ($request->filled('date') && $request->date !== 'Select Date') $filters['date'] = $request->date;
        if ($request->filled('keyword')) $filters['keyword'] = $request->keyword;
        if ($request->filled('delivery')) $filters['delivery'] = $request->delivery;
        if ($request->filled('features')) $filters['features'] = $request->features;

        if ($request->filled('date_filter') && $request->date_filter !== 'all') {
            $today = Carbon::today();

            if ($request->date_filter === 'today') {
                $filters['date'] = $today->format('d/m/Y');
            } elseif ($request->date_filter === '7days') {
                $filters['date_range'] = [
                    'from' => $today->format('d/m/Y'),
                    'to'   => $today->copy()->addDays(7)->format('d/m/Y'),
                ];
            } elseif ($request->date_filter === 'month') {
                $filters['date_range'] = [
                    'from' => $today->format('d/m/Y'),
                    'to'   => $today->copy()->endOfMonth()->format('d/m/Y'),
                ];
            }
        }

        $allEvents = empty($filters)
            ? $this->service->getAllEvents()
            : $this->service->flexibleSearchEvents($filters);

        if ($request->filled('price_range')) {
            $range = explode('-', $request->price_range);
            $min = $range[0] ?? 0;
            $max = $range[1] ?? PHP_INT_MAX;

            $allEvents = array_filter($allEvents, fn($e) =>
                ($e['minTicketPrice']['price'] ?? 0) >= $min &&
                ($e['minTicketPrice']['price'] ?? 0) <= $max
            );
            $allEvents = array_values($allEvents);
        }

        
        if ($request->filled('sort')) {
            if ($request->sort == 'price_asc') {
                usort($allEvents, fn($a, $b) => ($a['minTicketPrice']['price'] ?? 0) <=> ($b['minTicketPrice']['price'] ?? 0));
            } elseif ($request->sort == 'price_desc') {
                usort($allEvents, fn($a, $b) => ($b['minTicketPrice']['price'] ?? 0) <=> ($a['minTicketPrice']['price'] ?? 0));
            }
        }

        $dates = !empty($allEvents) ? $this->service->getUniqueEventDates($allEvents) : [];
        $cities = $request->filled('country_id')
            ? $this->service->getCities($request->country_id)['data'] ?? []
            : [];

        $page = Paginator::resolveCurrentPage() ?: 1;
        $perPage = 10;
        $items = array_slice($allEvents, ($page - 1) * $perPage, $perPage);
        $events = new LengthAwarePaginator($items, count($allEvents), $perPage, $page, [
            'path' => Paginator::resolveCurrentPath(),
            'query' => $request->query(),
        ]);

        return view('front.ticket', compact('countries', 'cities', 'dates', 'events', 'filters'));
    }

    public function ticket_show($id)
    {
        $event = $this->service->getEvent($id);
        $tickets = $this->service->getEventTickets($id);

        return view('front.ticket_show', compact('event', 'tickets'));
    }
}
