<?php

namespace App\Http\Controllers\Front;

use App\Http\Requests\Front\commentRequest;
use Illuminate\Support\Facades\Http;
use App\Models\BlogCategory;
use App\Models\CategoryFaq;
use App\Models\Comment;
use App\Models\Contest;
use App\Models\Course;
use App\Models\CourseChapter;
use App\Models\CourseFeature;
use App\Models\CourseReview;
use App\Models\Grade;
use App\Models\InstructorProfile;
use App\Models\Newsletter;
use App\Models\PageHeader;
use App\Models\Photo;
use App\Models\ResearchProject;
use App\Models\Slider;
use App\Models\ProductEncyclopedia;
use App\Models\About;
use App\Models\Product;

use App\Models\Partner;
use App\Models\Banner;
use App\Models\User;
use App\Models\Certificate;
use App\Models\Contact;
use App\Models\CountryCode;
use App\Models\ContactForm;
use App\Models\Blog;
use App\Models\Service;
use App\Models\Faq;
use App\Models\Setting;
use App\Models\GalleryCategory;
use App\Models\ProductCategory;
use App\Models\Tag;
use App\Models\Team;
use App\Mail\Mail;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use PhpOffice\PhpSpreadsheet\Calculation\Category;

class HomeController extends Controller
{

    public function index()
    {
        //slider
        $slider = Slider::whereHas('photo')->whereIn('type',['image','video'])->checkStatus()->orderBy('sort')->first();
        //ProductEncyclopedia
        $product_encyclopedias = ProductEncyclopedia::where('status_home', 'active')->checkStatus()->orderBy('sort')->take(6)->get();
        //about
        $about = About::where('type', 'home')->first();
        //product
        $products = Product::checkStatus()->orderByDesc('id')->take(12)->get();
        //about contact
        $about_contact = About::where('type', 'home_map')->first();
        //faq
        $faqs = Faq::where('status_home', 'active')->checkStatus()->orderBy('sort')->take(10)->get();
        //blogs
        $blogs = Blog::checkStatus()->orderByDesc('id')->take(10)->get();

        $partners = Partner::checkStatus()->orderByDesc('id')->take(15)->get();

        $services = Service::checkStatus()->orderByDesc('id')->get();

        $courses= Course::with(['instructors','groups','reviews'])->checkStatus()->orderByDesc('id')->get();

        $reviews=CourseReview::with(['user','course'])->where('parent_id',null)->orderByDesc('id')->get();

        $courses->each(function ($course){
            $sessions = $course->groups->first()->sessions ?? collect();

            $days = $sessions->pluck('day_of_week')->all();

            $time = $sessions->first()
                ?Carbon::parse($sessions->first()->start_time)->format('g:i A')
                . ' - ' . Carbon::parse($sessions->first()->end_time)->format('g:i A')
                : '';

            $course->schedule_string = count($days) ? implode(' and ', $days) . ', ' . $time : '';
        });

        return view('front.index', compact('slider','product_encyclopedias', 'about','products', 'about_contact', 'faqs', 'blogs','partners','services','courses','reviews'), ['title' => read_lang_word('title-home')]);
    }

    public function about($slug=null)
    {
        if (blank($slug))
        {
            $slug='about';
            $title=read_lang_word('title-about');
        }
        $item = About::where('type', str_replace('-','_',$slug))->first();
        $teams = Team::where('status', 1)->orderBy('sort')->get();
        $roles = Team::where('status',1)->pluck('text')->unique();
        $certificate=Certificate::checkStatus()->get();
        $hero=PageHeader::where('page','about')->first();

        if ($slug != 'about' && 'our_mission' && 'our_vision')
        {
            $title=read_lang($item,'title');
        }
        return view('front.about', compact(['item','teams','roles','certificate','hero']), ['title' =>  $title]);
    }

    public function certificate()
    {
        $items = Certificate::checkStatus()->get();
        return view('front.certificate', compact('items'), ['title' =>  read_lang_word('certificates')]);
    }
    public function faq()
    {
        $items = Faq::checkStatus()->orderBy('sort')->get();
        $category_faq = CategoryFaq::whereNull('parent_id')
            ->with(['children.faqs', 'faqs'])
            ->whereHas('faqs')
            ->orWhereHas('children.faqs')
            ->get();
        $categories=CategoryFaq::whereNull('parent_id')->with('children')->get();
        $hero=PageHeader::where('page','faq')->first();
        return view('front.faq.index', compact(['items','category_faq','categories','hero']), ['title' =>  read_lang_word('title-faq')]);
    }

    public function contact()
    {
        $item = Contact::first();
        $hero=PageHeader::where('page','contact')->first();
        return view('front.contact', compact(['item','hero']), ['title' =>  read_lang_word('title-contact')]);
    }

    public function contact_post(Request $request)
    {
            $this->validate($request, [
                'name' => 'required|max:200',
                'email' => 'nullable|email',
                'phone' => 'nullable|numeric',
                'subject' => 'required|max:250',
                'message' => 'required',

            ]);

        try {
            $item = ContactForm::create([
                'name' => $request->name,
                'email' => $request->email,
                'subject' => $request->subject,
                'message' => $request->message,
            ]);

            if ($request->hasFile('attachment')) {
                $photo = new Photo();
                $photo->type = 'attachment';
                $photo->path = file_store($request->attachment, 'assets/admin/uploads/contactus' . '/');
                $item->photo()->save($photo);
            }
            return redirect()->back()->with('success', read_lang_word('msg-success-form'));
           // return ['status'=>1 ,'message'=>read_lang_word('msg-success-form')];
        } catch (\Exception $e) {
            return ['status'=>0 ,'message'=>read_lang_word('msg-err-form')];
        }
    }

    public function service_list()
    {
        $items = Service::checkStatus()->orderBy('sort')->paginate(24);
        $tags=Tag::checkStatus()->orderBy('sort')->get();
        $categories=BlogCategory::checkStatus()->orderBy('sort')->get();
        return view('front.service.index', compact(['items','tags','categories']), ['title' => read_lang_word('title-service')]);
    }

    public function blog_list(Request $request)
    {
       // $items = Blog::checkStatus()->with('user','category')->orderBy('sort')->paginate(12);
        $query = Blog::checkStatus()->with('user','category');
        if( $request->filled('q'))
        {
            $search=$request->q;
            $query->where(function($q) use ($search) {
                $q->where('title', 'like', "%{$search}%")
                    ->orWhere('excerpt', 'like', "%{$search}%")
                    ->orWhere('text', 'like', "%{$search}%");
            });
        }
        //category
        if($request->filled('category'))
        {
            $categorySlug = $request->category;
            $category = BlogCategory::where('slug', $categorySlug)->first();
            if($category){
                $query->where('category_id', $category->id);
            }
        }
        //tags
        if($request->filled('tag'))
        {
            $tagId = $request->tag;
            $query->whereJsonContains('tags', $tagId);
        }
        //Reading Time Filter
        if($request->filled('reading'))
        {
            switch($request->reading){
                case 'Under 5 min':
                    $query->where('reading_time', '<', 5);
                    break;
                case '5–10 min':
                    $query->whereBetween('reading_time', [5,10]);
                    break;
                case '10+ min':
                    $query->where('reading_time', '>', 10);
                    break;
            }
        }
        $items = $query->orderBy('sort')->paginate(12)->appends($request->all());


        $tags=Tag::checkStatus()->orderBy('sort')->get();
        $categories=BlogCategory::checkStatus()->orderBy('sort')->get();
        $items_seen = Blog::checkStatus()->orderByDesc('seen')->take(5)->get();
        $colors = ['bg-red-500','bg-yellow-500','bg-purple-500','bg-pink-500','bg-blue-500'];
        $tagColors = [];
        $colorCount = count($colors);

        foreach($tags as $index => $tag) {
            $tagColors[$tag->id] = $colors[$index % $colorCount];
        }

        $blogs = Blog::checkStatus()->pluck('reading_time');

        if ($blogs->contains(fn($t) => $t < 5)) {
            $readingFilters[] = 'Under 5 min';
        }
        if ($blogs->contains(fn($t) => $t >= 5 && $t <= 10)) {
            $readingFilters[] = '5–10 min';
        }
        if ($blogs->contains(fn($t) => $t > 10)) {
            $readingFilters[] = '10+ min';
        }
        $hero=PageHeader::where('page','blog')->first();
        return view('front.blog.index', compact(['items','tags','categories','items_seen','tagColors','readingFilters','hero']), ['title' => read_lang_word('title-blog')]);
    }

    public function blog_show($slug)
    {

        $item = Blog::with('user','category')->where('slug', $slug)->checkStatus()->firstOrFail();
        $item->seen += 1;
        $item->update();
        //tag

        if(!empty($item->tags)){
            $tags=Tag::whereIN('id',$item->tags)->checkStatus()->get();
        }else{
            $tags=null;
        }
        $comments = Comment::where('blog_id', $item->id)
            ->whereNull('parent_id')
            ->with('replies', 'user')
            ->latest()
            ->get();
        //seen
        $tags=Tag::checkStatus()->orderBy('sort')->get();
        $categories=BlogCategory::checkStatus()->orderBy('sort')->get();
        $items_seen = Blog::checkStatus()->orderByDesc('seen')->take(5)->get();
        $colors = ['bg-red-500','bg-yellow-500','bg-purple-500','bg-pink-500','bg-blue-500'];
        $tagColors = [];
        $colorCount = count($colors);

        foreach($tags as $index => $tag) {
            $tagColors[$tag->id] = $colors[$index % $colorCount];
        }

        $blogs = Blog::checkStatus()->pluck('reading_time');

        if ($blogs->contains(fn($t) => $t < 5)) {
            $readingFilters[] = 'Under 5 min';
        }
        if ($blogs->contains(fn($t) => $t >= 5 && $t <= 10)) {
            $readingFilters[] = '5–10 min';
        }
        if ($blogs->contains(fn($t) => $t > 10)) {
            $readingFilters[] = '10+ min';
        }
        return view('front.blog.show', compact(['item', 'tags','categories', 'items_seen','tags','tagColors','readingFilters','comments']), ['title' => read_lang($item, 'title')]);
    }
    public function tag_show($slug)
    {
        $item = Tag::where('slug', $slug)->checkStatus()->firstOrFail();
        //last items
        $blogs = Blog::orderByDesc('id')->where('tags', 'LIKE', '%"'.$item->id.'"%')->checkStatus()->paginate('24');
        return view('front.tag.show', compact('item', 'blogs'), ['title' => read_lang($item, 'title')]);
    }

    public function product_list($slug = null)
    {
        $title=read_lang_word('products');
        $products=Product::checkStatus()->orderByDesc('id')->paginate(24);
        $item=null;
        if(!blank($slug))
        {
            $item=ProductCategory::where('slug',$slug)->checkStatus()->firstOrFail();
            $title=read_lang($item,'title');

            $products=Product::where('category_id',$item->id)->checkStatus()->orderByDesc('id')->paginate(24);
        }
        $breadcrumb=[];
        array_push($breadcrumb,['title'=>read_lang_word('products') , 'url'=>route('front.product.list')]);
        if($item)
        {
            if($item->parent)
            {
                array_push($breadcrumb,['title'=>read_lang($item->parent,'title') , 'url'=>route('front.product.list',$item->parent->slug)]);
            }
            array_push($breadcrumb,['title'=>read_lang($item,'title') , 'url'=>route('front.product.list',$item->slug)]);
        }

        return view('front.product.index', compact('products','item'), ['title' => $title,'breadcrumb'=>$breadcrumb]);
    }
    public function product_show($slug)
    {
        $item=Product::checkStatus()->where('slug',$slug)->firstOrFail();
        $cat=ProductCategory::findOrFail($item->category_id);

        $products=Product::where('id','!=',$item->id)->where('category_id',$item->category_id)->checkStatus()->orderByDesc('id')->take(18)->get();
        $breadcrumb=[];
        array_push($breadcrumb,['title'=>read_lang_word('products') , 'url'=>route('front.product.list')]);
        if($cat)
        {
            array_push($breadcrumb,['title'=>read_lang($cat,'title') , 'url'=>route('front.product.list',$cat->slug)]);
        }
        array_push($breadcrumb,['title'=>read_lang($item,'title') , 'url'=>route('front.product.show',$item->slug)]);

        return view('front.product.show', compact('item','products'), ['title' => read_lang($item,'title'),'breadcrumb'=>$breadcrumb]);
    }
    public function encyclopedia_list()
    {
        $items=ProductEncyclopedia::checkStatus()->orderBy('sort')->take(6)->get();

        return view('front.encyclopedia.index', compact('items'), ['title' => read_lang_word('encyclopedia')]);
    }
    public function encyclopedia_show($slug)
    {
        $item=ProductEncyclopedia::checkStatus()->where('slug',$slug)->firstOrFail();
        $products=Product::whereIN('id',json_decode($item->product_id))->checkStatus()->get();
        $similar_items=ProductEncyclopedia::checkStatus()->take(10)->get();
        $breadcrumb=[];
        array_push($breadcrumb,['title'=>read_lang_word('encyclopedia') , 'url'=>route('front.encyclopedia.list')]);
        if($item)
        {
            array_push($breadcrumb,['title'=>read_lang($item,'title') , 'url'=>route('front.encyclopedia.list',$item->slug)]);
        }
        return view('front.encyclopedia.show', compact('item','products','similar_items'), ['title' => read_lang($item,'title'),'breadcrumb'=>$breadcrumb]);
    }

    public function gallery()
    {
        $items = GalleryCategory::checkStatus()->orderBy('sort')->paginate(24);
        return view('front.gallery.index', compact('items'), ['title' => read_lang_word('title-gallery')]);
    }

    public function research_list()
    {

       $title=read_lang_word('research-project');

        $projects=ResearchProject::where('status', 'active')->orderBy('sort')->get();
        $item=About::where('type', 'research_Development')->first();
        $research=About::where('type', 'research')->first();
        $hero=PageHeader::where('page','research')->first();
        return view('front.research', compact(['projects','item','research','hero']), ['title' =>  $title]);
    }


    public function newsletter(Request $request)
    {
        $this->validate($request, [
            'email' => 'nullable|email',
        ]);

        try {
            $item = Newsletter::create([
                'email' => $request->email,
            ]);

            return redirect()->back()->with('success-newsletter', read_lang_word('msg-success-form'));
            // return ['status'=>1 ,'message'=>read_lang_word('msg-success-form')];
        } catch (\Exception $e) {
            dd($e->getMessage());
            return ['status'=>0 ,'message'=>read_lang_word('msg-err-form')];
        }
    }

    public function send_comment(commentRequest $request,Blog $blog)
    {

        $recaptchaResponse = $request->input('g-recaptcha-response');

        $response = Http::asForm()->post('https://www.google.com/recaptcha/api/siteverify', [
            'secret' => env('RECAPTCHA_SECRET_KEY'),
            'response' => $recaptchaResponse,
            'remoteip' => $request->ip(),
        ]);

        $responseData = $response->json();

        if (! $responseData['success']) {
            return back()->withErrors(['captcha' => read_lang_word('robot-failed')]);
        }

        if(empty($request->parent_id))
        {
            $existing=Comment::where( 'user_id',Auth::id())->where('blog_id', $blog->id)->whereNull('parent_id') ->first();
            if ($existing) {
                return redirect()->back()->with('error', read_lang_word('already-commented'));
            }
        }
        Comment::create([
            'user_id' => Auth::id(),
            'blog_id' => $blog->id,
            'parent_id' => $request->parent_id,
            'description' => $request->description,
        ]);
        return redirect()->back()->with('success', read_lang_word('success-form'));
    }

    public function course_list()
    {

        $courses= Course::with(['instructors','groups','reviews','category','grade'])->checkStatus()->orderByDesc('id')->paginate(12);
        $months = $courses->map(function($course) {
            return $course->start_date ? \Carbon\Carbon::parse($course->start_date)->format('F') : null;
        })->filter()->unique()->values();
        $categories=CategoryFaq::whereNull('parent_id')->where('title', 'Subject Area')->with('children')->get();
        $contests=Contest::select('id','name','icon')->get();
        $grades=Grade::select('id','name','slug')->get();
        $hero=PageHeader::where('page','course')->first();
        return view('front.course.index',compact(['courses','categories','grades','months','contests','hero']));
    }

    public function course_show($slug)
    {
        $item = Course::with([
            'instructors.instructorProfile',
            'instructors.teachingCourses',
            'groups',
            'reviews',
            'category',
            'grade',
            'contest'
        ])
            ->where('slug', $slug)
            ->checkStatus()
            ->firstOrFail();

        $item->reviews_count +=1;
        $item->update();

        $chapters = CourseChapter::where('course_id', $item->id)->get();
        $faqs = Faq::where('category_id', $item->category_id)->get();

        $teachers = $item->instructors;

        $relatedCourses = Course::where('id', '!=', $item->id)
            ->where(function($q) use ($item) {
                $q->where('category_id', $item->category_id)
                    ->orWhere('level', $item->level)
                    ->orWhereHas('instructors', function($q2) use ($item) {
                        $q2->whereIn('users.id', $item->instructors->pluck('id'));
                    });
            })
            ->checkStatus()
            ->inRandomOrder()
            ->take(4)
            ->get();

        $sessions = $item->groups->first()->sessions ?? collect();

        $days = $sessions->pluck('day_of_week')->all();

        $time = $sessions->first()
            ? Carbon::parse($sessions->first()->start_time)->format('g:i A')
            . ' - ' . Carbon::parse($sessions->first()->end_time)->format('g:i A')
            : '';

        $item->schedule_string = count($days)
            ? implode(' and ', $days) . ', ' . $time
            : '';
        $count = $item->instructors->count();
        $comments = CourseReview::where('course_id', $item->id)
            ->whereNull('parent_id')
            ->with('replies', 'user')
            ->latest()
            ->get();
        return view('front.course.course', compact(['item','chapters','faqs','relatedCourses','count','comments']));
    }

    public function send_comment_course(commentRequest $request,$course)
    {
        try {

            $recaptchaResponse = $request->input('g-recaptcha-response');

            $response = Http::asForm()->post('https://www.google.com/recaptcha/api/siteverify', [
                'secret' => env('RECAPTCHA_SECRET_KEY'),
                'response' => $recaptchaResponse,
                'remoteip' => $request->ip(),
            ]);

            $responseData = $response->json();

            if (! $responseData['success']) {
                return back()->withErrors(['captcha' => read_lang_word('robot-failed')]);
            }

            if(empty($request->parent_id))
            {
                $existing=CourseReview::where( 'user_id',Auth::id())->where('course_id', $course)->whereNull('parent_id') ->first();
                if ($existing) {
                    return redirect()->back()->with('error', read_lang_word('already-commented'));
                }
            }
            CourseReview::create([
                'user_id'   => Auth::id(),
                'course_id' => $course,
                'parent_id' => $request->parent_id,
                'description'   => $request->description,
            ]);

            return redirect()->back()->with('success', read_lang_word('success-form'));

        } catch (\Exception $e) {
             return redirect()->back()->with('error',read_lang_word('err-message'));
        }
    }

}
