<?php

namespace App\Http\Controllers\Admin\Setting;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Services\TranslationService;
use App\Models\Translation;
use Illuminate\Support\Str;

class TranslationController extends Controller
{
    public function index()
    {
        $items = Translation::orderByDesc('id')->get();
        return view('admin.setting.translations.index', compact('items'), ['title' => 'مترجم پنل']);
    }

    public function extractAndTranslate(TranslationService $service)
    {
        try {
            $count = $service->extractAndTranslateAllAdminPages();

            return redirect()->route('admin.translations.index')->with('flash_message', 'Bilgiler başarıyla eklendi.');
        } catch (\Exception $e) {
            return redirect()->route('admin.translations.index')
                ->with('err_message', 'خطا در استخراج متون: ' . $e->getMessage());
        }
    }

    public function store(Request $request)
    {
        try {
            $translation = Translation::where('key', $request->key)->first();
            if (!$translation) {
                $translation = Translation::create([
                    'key' => $request->key
                ]);
            }

            foreach (['en', 'fr'] as $lang) {
                if (!isset($translation->translations[$lang])) {
                    $translated = app(\App\Services\TranslationService::class)->translateText($request->key, 'fa', $lang);
                    $translation->addTranslation($lang, $translated);
                }
            }

            return redirect()->back()->with('flash_message', 'Bilgiler başarıyla eklendi.');
        } catch (\Exception $e) {
            return redirect()->back()->withInput()->with('err_message', 'Listeye eklemede bir sorunla karşılaştık, lütfen tekrar deneyin.');
        }
    }

    public function update(Request $request, $id)
    {
        try {
            $translation = \App\Models\Translation::findOrFail($id);

            $translations = $translation->translations;
            foreach ($request->translations as $lang => $text) {
                if (!empty($text)) {
                    $translations[$lang] = $text;
                }
            }

            $translation->update(['translations' => $translations]);

            return redirect($this->controller_title('url_back'))->with('flash_message', 'Bilgiler başarıyla düzenlendi.');
        } catch (\Exception $e) {
            return redirect()->back()->withInput()->with('err_message', 'Düzenlemede sorun yaşadık, lütfen tekrar deneyin.');
        }
    }


    public function processAjax(Request $request, TranslationService $translationService)
    {
        $offset = (int) $request->input('offset', 0);
        $perPage = (int) $request->input('per_page', 5);

        // جمع‌آوری مسیرهای admin
        $allRoutes = collect(\Route::getRoutes())
            ->filter(function ($route) {
                $isAdmin = false;
                if (isset($route->action['prefix'])) {
                    $isAdmin = Str::startsWith($route->action['prefix'], 'admin');
                }
                if (!$isAdmin && isset($route->action['as'])) {
                    $isAdmin = Str::startsWith($route->action['as'], 'admin.');
                }
                if (!$isAdmin) {
                    $isAdmin = Str::startsWith($route->uri(), 'admin/');
                }
                return $isAdmin && in_array('GET', $route->methods());
            })
            ->map(function ($route) {
                return $route->uri();
            })
            ->unique()
            ->values();

        $total = $allRoutes->count();

        if ($offset >= $total) {
            return response()->json([
                'status' => 'completed',
                'message' => 'تمام صفحات ادمین پردازش شدند',
                'processed' => $total,
                'total' => $total
            ]);
        }

        $routesToProcess = $allRoutes->slice($offset, $perPage);
        $processedCount = 0;

        foreach ($routesToProcess as $route) {
            try {
                // استفاده از درخواست داخلی (GET)
                $req = \Illuminate\Http\Request::create($route, 'GET');
                $resp = app()->handle($req);

                if ($resp->isSuccessful()) {
                    $content = $resp->getContent();
                    if (!empty($content)) {
                        $texts = $translationService->extractAdminTexts($content);
                        $this->processTexts($texts);
                        $processedCount++;
                    }
                } else {
                    // اگر redirect یا 403 بود، نادیده بگیر
                    \Log::info("Route {$route} returned status " . $resp->getStatusCode());
                }
            } catch (\Exception $e) {
                \Log::warning("Error processing route {$route}: ".$e->getMessage());
                continue;
            }
        }

        $newOffset = $offset + $processedCount;

        return response()->json([
            'status' => $newOffset >= $total ? 'completed' : 'processing',
            'message' => 'در حال پردازش صفحات ادمین',
            'processed' => $newOffset,
            'total' => $total,
            'next_offset' => $newOffset
        ]);
    }


    private function processTexts(array $texts)
    {
        $texts = array_unique(array_filter($texts, [app(\App\Services\TranslationService::class), 'isValidText']));

        foreach ($texts as $text) {
            $translation = \App\Models\Translation::findOrCreate($text);

            foreach (['en', 'tr', 'ar'] as $lang) {
                if (!isset($translation->translations[$lang])) {
                    $translated = app(\App\Services\TranslationService::class)->translateText($text, 'fa', $lang);
                    $translation->addTranslation($lang, $translated);
                }
            }
        }
    }

}
