<?php

namespace App\Http\Controllers\Admin\Setting;

use App\Models\Redirect;
use App\Http\Requests\Setting\RedirectRequest;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\File;

use Spatie\Permission\Models\Permission;

class RedirectController extends Controller
{
    public function controller_title($type)
    {
        switch ($type) {
            case 'index':
                return 'ریدایرکت';
                break;
            case 'create':
                return 'افزودن  ریدایرکت';
                break;
            case 'edit':
                return 'ویرایش ریدایرکت';
                break;
            case 'url_back':
                return route('admin.redirect.index');
                break;
            default:
                return '';
                break;
        }
    }

    public function __construct()
    {
        $this->middleware('permission:redirect_list', ['only' => ['index', 'show']]);
        $this->middleware('permission:redirect_create', ['only' => ['create', 'store']]);
        $this->middleware('permission:redirect_edit', ['only' => ['edit', 'update']]);
        $this->middleware('permission:redirect_delete', ['only' => ['destroy']]);
        $this->middleware('permission:redirect_status', ['only' => ['status']]);
    }

    public function index()
    {
        $items = Redirect::orderByDesc('id')->get();
        return view('admin.setting.redirect.index', compact('items'), ['title' => $this->controller_title('index')]);
    }

    public function show($id)
    {

    }

    public function create(Request $request)
    {
        $url_back = $this->controller_title('url_back');
        $type = $request->type == 410 ? $request->type : 301;
        return view('admin.setting.redirect.create', compact('url_back', 'type'), ['title' => $this->controller_title('create')]);
    }

    public function store(RedirectRequest $request)
    {
        if(!in_array($request->type,[410,404,301,302]))
        {
            abort(404);
        }
        try {
            $item = Redirect::create([
                'title' => $request->title,
                'from' => urldecode($request->from),
                'to' =>  in_array($request->type,[301,302]) ? urldecode($request->to) : null,
                'groups' =>  in_array($request->type,[301,302]) ? $request->groups : 'no',
                'type' => $request->type,
                'status' => $request->status,
                'user_id_create' => auth()->id(),
            ]);
            return redirect($this->controller_title('url_back'))->with('flash_message', 'اطلاعات با موفقیت افزوده شد');
        } catch (\Exception $e) {
            return redirect()->back()->withInput()->with('err_message', 'برای افزودن به مشکل خوردیم، مجدد تلاش کنید');
        }
    }

    public function edit($id)
    {
        $url_back = $this->controller_title('url_back');
        $item = Redirect::findOrFail($id);
        return view('admin.setting.redirect.edit', compact('url_back', 'item'), ['title' => $this->controller_title('edit')]);
    }

    public function update(RedirectRequest $request, $id)
    {
        $item = Redirect::findOrFail($id);
        if(!in_array($request->type,[410,404,301,302]))
        {
            abort(404);
        }
        try {
            Redirect::where('id', $id)->update([
                'title' => $request->title,
                'from' => urldecode($request->from),
                'to' => in_array($request->type,[301,302]) ? urldecode($request->to) : null,
                'type' => $request->type,
                'groups' =>  in_array($request->type,[301,302]) ? $request->groups : 'no',
                'status' => $request->status,
                'user_id_edit' => auth()->id(),
            ]);
            return redirect($this->controller_title('url_back'))->with('flash_message', 'اطلاعات با موفقیت ویرایش شد');
        } catch (\Exception $e) {
            return redirect()->back()->withInput()->with('err_message', 'برای ویرایش به مشکل خوردیم، مجدد تلاش کنید');
        }
    }

    public function destroy($id)
    {
        $item = Redirect::findOrFail($id);
        try {
            $item->delete();
            return redirect($this->controller_title('url_back'))->with('flash_message', 'اطلاعات با موفقیت حذف شد');
        } catch (\Exception $e) {
            return redirect()->back()->withInput()->with('err_message', 'برای حذف به مشکل خوردیم، مجدد تلاش کنید');
        }
    }

    public function status($id, $status)
    {
        $item = Redirect::findOrFail($id);
        try {
            $item->status = $status;
            $item->update();
            return redirect($this->controller_title('url_back'))->with('flash_message', 'اطلاعات با موفقیت تغیر وضعیت شد');
        } catch (\Exception $e) {
            return redirect()->back()->withInput()->with('err_message', 'برای تغیر وضعیت به مشکل خوردیم، مجدد تلاش کنید');
        }
    }
}
